
  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Tue Nov 25 10:17:10 2007
   *  Last modified   :  %modify_time%
   */
  /** @file 
   *  \brief A brief description of this module 
   *
   *  \version CRYS_ATP_TST_HMACFLOW.c#1:csrc:1
   *  \author Annag
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */
/************* Include Files ****************/
#include "CRYS.h"

#include "CRYS_ATP_TST_Defines.h"
#include "CRYS_ATP_TST_UTIL_Funcs.h"
#include "CRYS_ATP_TST_UTIL_OTF_Funcs.h"
#include "CRYS_ATP_TST_HMACFLOW_Acceptance.h"
#include "CRYS_ATP_TST_AESFLOW_Acceptance.h"
#include "CRYS_ATP_TST_DESFLOW_Acceptance.h"
#include "CRYS_ATP_TST_HASHFLOW_Acceptance.h"
#include "CRYS_ATP_TST_HMACFLOW_Acceptance_test_vectors.h"
#include "CRYS_ATP_TST_FLOW_Defines.h"
#include "gen.h"
#include "CRYS_INIT_TST.h"
#include "CRYS_FLOW.h"
#include "CRYS_FLOW_error.h"
#include "DX_VOS_Mem.h"
#include "FLOW_error.h"

#if TST_USE_FLOW_DATA


/************************ Defines *******************************/
/************************ Enums *********************************/
/************************ Typedefs ******************************/
/************************ Global Data ***************************/

extern DxUint8_t 	                       TST_Input_Data[TST_FLOW_DATA_VEC_SIZE_IN_BYTS];
extern DxUint64_t	                       TST_FLOW_InputData[TST_FLOW_DATA_VEC_SIZE_IN_BYTS/8];

extern DxUint64_t						   TST_FLOW_OutputData[TST_FLOW_NUMBER_OF_SUPPORTED_ENGINES][TST_FLOW_DATA_VEC_SIZE_IN_BYTS/8];

/* common data for one test vector */
extern TST_FLOW_TestCommonDataStruct       TST_TestData;

/* input data size */
extern DxUint32_t                          TST_FLOW_InputDataSize;
extern CRYS_FLOW_Params_t                  TST_FLOWParamsInfo;

/* output data of HASH/AES_MAC result */
extern DxUint32_t                          TST_FLOW_Hash_OutputData[CRYS_HASH_RESULT_SIZE_IN_WORDS];

/* output data for integrated engines */
extern DxUint64_t	                       TST_FLOW_OutputDataForIntegratedEngines[TST_FLOW_DATA_VEC_SIZE_IN_BYTS/8];

/* indication if data size in function finish should be zero */
extern DxBool_t                            TST_FLOW_IsFinishSizeZero;

/* flow input buffers */
extern CRYS_FLOW_BufferData_t              TST_InputBuffers[TST_FLOW_DATA_MAX_NUM_OF_FLOWBUFFERS];

/* flow output buffers */
extern CRYS_FLOW_BufferData_t              TST_OutputBuffers[TST_FLOW_NUMBER_OF_SUPPORTED_ENGINES][TST_FLOW_DATA_MAX_NUM_OF_FLOWBUFFERS];

/* HMAC engine data */
CRYS_FLOW_HMACEngine_Data_t                TST_FLOWHMACEngineData;


/************* Private function prototype ***********************/


/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_InitHMACEngineData
 *  
 * Inputs:
 *  TST_index - test index
 *  None
 *
 * Outputs:                                 
 *
 * Description:                                                                 
 *  This function initializes engine data for HMAC FLOW operation
 * Algorithm:     
 *	This function initializes HMAC flow engine data struct with values from the test vector.
 ******************************************************************************/                                
void CRYS_ATP_TST_FLOW_InitHMACEngineData ( int TST_index ) 
{
    
    /* initialize key value */
    TST_FLOWHMACEngineData.Key_ptr = TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Key;
    TST_FLOWHMACEngineData.KeySize = TST_HMACFLOW_Acceptance_TestData[TST_index].TST_KeySize; 
    TST_FLOWHMACEngineData.opMode  = TST_HMACFLOW_Acceptance_TestData[TST_index].TST_OperationMode; 
}

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_InitHMACFlow
 *  
 * Inputs:
 *  TST_index          - test index
 *  flowType           - flow type
 *  TST_MasterTestName - test name
 *  TST_HMACFlowId      - output HMAC flow id
 *  pHMACFlowParamsInfo - flow params info
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *	This function starts HMAC flow.
 *
 * Algorithm:     
 *	This function inits HMAC flow.
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_InitHMACFlow ( int                     TST_index,
                                        CRYS_FLOW_Type_t        flowType,
                                        char                   *TST_MasterTestName, 
                                        DxUint32_t             *TST_HMACFlowId,
                                        CRYS_FLOW_Params_t     *pHMACFlowParamsInfo ) 
{
    /********* TST Variables Definitions ************/
    CRYSError_t  TST_TestStatus = TESTPASS;
    
    TST_TestData.inputBigEndianOrderFlag  = TST_HMACFLOW_Acceptance_TestData[TST_index].inputBigEndianOrderFlag;
	TST_TestData.inputBuffersArraySize    = TST_HMACFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
	TST_TestData.inputInterruptId         = TST_HMACFLOW_Acceptance_TestData[TST_index].inputInterruptId;
	TST_TestData.outputBigEndianOrderFlag = TST_HMACFLOW_Acceptance_TestData[TST_index].outputBigEndianOrderFlag;
	TST_TestData.outputBuffersArraySize   = TST_HMACFLOW_Acceptance_TestData[TST_index].outputBuffersArraySize;
	TST_TestData.outputInterruptId        = TST_HMACFLOW_Acceptance_TestData[TST_index].outputInterruptId;
	TST_TestData.wordsBuffersOrderFlag    = TST_HMACFLOW_Acceptance_TestData[TST_index].wordsBuffersOrderFlag;
	
    
    /* initialize flow init parameters */
 	CRYS_ATP_TST_FLOW_InitParams(TST_index, DX_TRUE, &TST_TestData, pHMACFlowParamsInfo, CRYS_FLOW_HMAC);
 	    
 	/* initialize HMAC engine data */
 	CRYS_ATP_TST_FLOW_InitHMACEngineData(TST_index);
         	
    
    /* Init HMAC flow */
 	/***********************/
 	TST_ATP_Error = CRYS_FLOW_HMACInit(flowType,
                 	                   pHMACFlowParamsInfo,
                                       &TST_FLOWHMACEngineData,  
                                       TST_HMACFlowId);
                      
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_HMACInit is done\n"));
                        
    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_HMACInit",TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
    
    #if CRYS_ATP_TST_EXIT_ON_ERROR
        if (TST_TestStatus != TESTPASS) 
            return TESTFAIL;
    #endif  
    
    return TST_TestStatus;
} /* CRYS_ATP_TST_InitHMACFlow */

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FinishHMACFlow
 *  
 * Inputs:
 *  TST_index           - test index
 *  TST_HMAC            - flow id
 *  TST_MasterTestName  - test name
 *  blockSize           - block size for finish
 *  pHMACFlowParamsInfo - flow params info
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *	This function finishes HMAC flow and check results.
 *
 * Algorithm:     
 *	1. Finish HMAC flow
 *  2. Get results
 *  3. Compare output buffer with expected results
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FinishHMACFlow ( int                     TST_index, 
                                          DxUint32_t              TST_HMACFlowId, 
                                          char                   *TST_MasterTestName,
                                          DxUint32_t              blockSize,
                                          CRYS_FLOW_Params_t     *pHMACFlowParamsInfo ) 
{
    CRYSError_t  TST_TestStatus = TESTPASS;
    
    
    /* Finish HMAC flow with size of the last input buffer */
    /******************************************************/
    TST_ATP_Error = CRYS_FLOW_Finish(TST_HMACFlowId,
                                     blockSize);  
                     
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Finish is done\n"));                             
    
    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Finish",TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

    #if CRYS_ATP_TST_EXIT_ON_ERROR
        if (TST_TestStatus != TESTPASS) 
            return TESTFAIL;
    #endif  
                     
    /* Get results */
	/***************/
	do
    {
        
        TST_ATP_Error = CRYS_FLOW_GetResults(TST_HMACFlowId,
		                                 (DxUint8_t*)TST_FLOW_Hash_OutputData,
		                                 CRYS_HASH_RESULT_SIZE_IN_WORDS*4);
	
    }while(TST_ATP_Error == DX_SEP_FLOW_GET_RESULT_NOT_READY_ERROR);
    
	                        
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_GetResults is done\n"));                             
	
	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_GetResults",TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  


	/* Compare output to expected result */
	/*************************************/
	/* compare and prints results of test */
	TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(
		(DxUint8_t*)TST_FLOW_Hash_OutputData,
		TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Expected_output_data,
		TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Expected_dataSize);


	/*Print the outputted and then the expected result after mem compare */
	ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
		"\nData outputted by engine [output buffer]-- \n",
			(DxUint8_t*)TST_FLOW_Hash_OutputData, 
		    TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Expected_dataSize); 

	ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
		"\nData Expected by engine --\n",
		TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Expected_output_data,
		TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Expected_dataSize);                 	                                     			
	

	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Name,TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  

    return TST_TestStatus;
}


/*****************************************************************************
* Function Name:                                                           
*  CRYS_ATP_TST_FLOW_CheckHMACOperation
*  
* Inputs:
*  TST_index - test index,
*  TST_MasterTestName - test name
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function checks some HMAC operation
* Algorithm:     
* 1. Execute some HMAC operation 
* 2. Compare result to expected one
******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_CheckHMACOperation () 
{
    /********* TST Variables Definitions ************/
    CRYSError_t  TST_TestStatus = TESTPASS;
    DxUint32_t   ExpectedSize = TST_FLOW_InputDataSize;
    /* use test index = 0 */
	DxUint32_t   TST_index = 0;
		      
    CRYS_HMAC(TST_HMACFLOW_Acceptance_TestData[0].TST_OperationMode,
              TST_HMACFLOW_Acceptance_TestData[0].TST_Key,
              TST_HMACFLOW_Acceptance_TestData[0].TST_KeySize,
              (DxUint8_t*)TST_Input_Data,
              TST_FLOW_InputDataSize,
              TST_FLOW_Hash_OutputData);
             
             
	/* compare HMAC result with one expected */
	/****************************************/
	TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(
		TST_FLOW_Hash_OutputData,
		TST_HMACFLOW_Acceptance_TestData[0].TST_Expected_output_data,
		ExpectedSize);					      

	ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CheckHMACOperation is done\n"));
    
    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CheckHMACOperation",TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Name,TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Name);

    #if CRYS_ATP_TST_EXIT_ON_ERROR
        if (TST_TestStatus != TESTPASS) 
            return TESTFAIL;
    #endif  
	return TST_ATP_Error;

}


/************************ Public Functions ******************************/

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_HMAC_ONETEST_Acceptance
 *  
 * Inputs:
 *  TST_index          - test index
 *  TST_MasterTestName - test name
 *  flowType           - flow type (static or dynamic)
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for static HMAC FLOW API's
 
 * Algorithm:     
 *  1. Initialize HMAC flow with type and other parameters
 *  2. Send interrupts that input and output buffers are ready
 *  3. Finish HMAC flow and compare output to expected results
 *  4. Terminate flow
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_HMAC_ONETEST_Acceptance ( int             TST_index, 
                                                        char            *TST_MasterTestName,
                                                        CRYS_FLOW_Type_t flowType ) 
{
    /********* TST Variables Definitions ************/
	CRYSError_t             	TST_TestStatus = TESTPASS;
	
	/* flow id */
	DxUint32_t                   flowId = 0;
	DxUint32_t                   i = 0;
	DxUint16_t                   blockSize;
	

	/************************************************/
	/*********** Start of test **********************/
	/************************************************/
	
    /* input block size */
    blockSize       = TST_FLOW_InputDataSize/TST_HMACFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
	
    /* initialize input buffer: */
    DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_Input_Data, TST_FLOW_InputDataSize);
	
	
	/* Initialize HMAC flow parameters and init HMAC flow by calling function */
    /* CRYS_FLOW_HMACInit                                                     */
	/**************************************************************************/
	TST_TestStatus = CRYS_ATP_TST_InitHMACFlow(TST_index, flowType, TST_MasterTestName, &flowId, &TST_FLOWParamsInfo);
	
	#if CRYS_ATP_TST_EXIT_ON_ERROR
    if (TST_TestStatus != TESTPASS) 
        return TESTFAIL;
    #endif  

	
	/* Send interrupts that input and output buffers are ready */
	/***********************************************************/
	

	/* Send interrupt that input buffers are ready */
	if (TST_FLOW_IsFinishSizeZero == DX_TRUE)
	{
	    for (i = 0; i < TST_FLOWParamsInfo.inputBuffersArraySize; ++i)
	        CRYS_ATP_TST_FLOW_SendInterrupt(TST_FLOWParamsInfo.inputInterruptId);
	        
        blockSize = 0;		        
	}
	else
	{
	    for (i = 0; i < TST_FLOWParamsInfo.inputBuffersArraySize - 1; ++i)
	        CRYS_ATP_TST_FLOW_SendInterrupt(TST_FLOWParamsInfo.inputInterruptId);
	}
		

	/* Finish HMAC flow, get results and compare results with expected  */
    /* by calling functions CRYS_FLOW_Finish and CRYS_FLOW_GetResults   */
    /********************************************************************/
	TST_TestStatus = CRYS_ATP_TST_FinishHMACFlow(TST_index, flowId, TST_MasterTestName, blockSize, &TST_FLOWParamsInfo);
	
	#if CRYS_ATP_TST_EXIT_ON_ERROR
    if (TST_TestStatus != TESTPASS) 
        return TESTFAIL;
    #endif  
	

    /* terminate flow */
    /******************/
	TST_ATP_Error = CRYS_FLOW_Terminate(flowId);
	
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Terminate is done\n"));                             
	
	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Terminate",TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  

	return TST_TestStatus;    
}/* End of CRYS_ATP_TST_FLOW_HMAC_ONETEST_Acceptance*/ 

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_HMACSTATIC_Acceptance
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for static HMAC FLOW API's
 
 * Algorithm:     
 *	This function goes through all tests vectors and execute HMAC static flow acceptance test
 *  for each vector
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_HMACSTATIC_Acceptance ( void ) 
{

	/********* TST Variables Definitions ************/
	int                     	TST_index;
	CRYSError_t             	TST_TestStatus = TESTPASS;
	char                     	TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_HMACSTATIC_Acceptance"; 
	
	
	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;
	TST_FLOWHMACEngineData.Key_ptr          = DX_NULL;


	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n"));
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_HMACSTATIC_Acceptance Tests ********************* \n"));      
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n \n"));


	/* go through all tests vectors and execute DES flow acceptance test for each vector */
    /*************************************************************************************/
	for (TST_index =0 ; TST_index < TST_NumberOf_HMACFLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
	{
		TST_TestStatus = CRYS_ATP_TST_FLOW_HMAC_ONETEST_Acceptance(TST_index, TST_MasterTestName, CRYS_FLOW_TypeStatic);
		

#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif  

	}/* End of TST_index */     

	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_HMACSTATIC_Acceptance*/ 

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_HMACDYNAMIC_Acceptance
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for static HMAC FLOW API's
 
 * Algorithm:     
 *	This function goes through all tests vectors and execute HMAC dynamic flow acceptance test
 *  for each vector
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_HMACDYNAMIC_Acceptance ( void ) 
{

	/********* TST Variables Definitions ************/
	int                     	TST_index;
	CRYSError_t             	TST_TestStatus = TESTPASS;
	char                     	TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_HMACDYNAMIC_Acceptance"; 
	
	
	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;
	TST_FLOWHMACEngineData.Key_ptr          = DX_NULL;


	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n"));
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_HMACDYNAMIC_Acceptance Tests ********************* \n"));      
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n \n"));


	/* go through all tests vectors and execute DES flow acceptance test for each vector */
    /*************************************************************************************/
	for (TST_index =0 ; TST_index < TST_NumberOf_HMACFLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
	{
		TST_TestStatus = CRYS_ATP_TST_FLOW_HMAC_ONETEST_Acceptance(TST_index, TST_MasterTestName, CRYS_FLOW_TypeDynamic);
		

#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif  

	}/* End of TST_index */     

	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_HMACDYNAMIC_Acceptance*/


/*****************************************************************************
* Function Name:                                                           
*  CRYS_ATP_TST_FLOW_HMACDYNAMIC_AddBuffers_Acceptance
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for dynamic HMAC FLOW API's with add buffers

* Algorithm:     
*  1. Initialize HMAC flow with dynamic type and other parameters
*  2. Init HMAC flow
*  3. Send interrupts that input and output buffers are ready
*  4. Add buffers to dynamic flow
*  5. Send interrupts that added input and output buffers are ready
*  6. Finish HMAC flow
*  7. Get results
*  8. Compare output to expected results
******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_HMACDYNAMIC_AddBuffers_Acceptance ( void ) 
{

	/********* TST Variables Definitions ************/
	int                     	 TST_index;
	CRYSError_t             	 TST_TestStatus = TESTPASS;
	char                     	 TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_HMACDYNAMIC_AddBuffers_Acceptance"; 
	
	/* flow id */
	DxUint32_t                   flowId = 0;
	int                          j = 0; 
	int                          inputBuffersArraySize = 1;

	CRYS_FLOW_BufferData_t       inputBuffersArray_ptr[10];
    DxUint16_t                   blockSize;
    
	

	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;

	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n"));
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_HMACDYNAMIC_AddBuffers_Acceptance Tests ********************* \n"));      
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n \n"));


	/************************************************/
	/*********** Start of test **********************/
	/************************************************/
	for (TST_index =0 ; TST_index < TST_NumberOf_HMACFLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
	{
		/* Initialize HMAC flow with dynamic type and other parameters */
        /**************************************************************/
		TST_TestData.inputBigEndianOrderFlag  = TST_HMACFLOW_Acceptance_TestData[TST_index].inputBigEndianOrderFlag;
		TST_TestData.inputBuffersArraySize    = TST_HMACFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
		TST_TestData.inputInterruptId         = TST_HMACFLOW_Acceptance_TestData[TST_index].inputInterruptId;
		TST_TestData.outputBigEndianOrderFlag = TST_HMACFLOW_Acceptance_TestData[TST_index].outputBigEndianOrderFlag;
		TST_TestData.outputBuffersArraySize   = TST_HMACFLOW_Acceptance_TestData[TST_index].outputBuffersArraySize;
		TST_TestData.outputInterruptId        = TST_HMACFLOW_Acceptance_TestData[TST_index].outputInterruptId;
		TST_TestData.wordsBuffersOrderFlag    = TST_HMACFLOW_Acceptance_TestData[TST_index].wordsBuffersOrderFlag;

		blockSize       = TST_FLOW_InputDataSize/TST_HMACFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
	

		DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_Input_Data, TST_FLOW_InputDataSize);
		
		/* initialize flow init parameters */
		CRYS_ATP_TST_FLOW_InitParams(TST_index, DX_FALSE, &TST_TestData, &TST_FLOWParamsInfo, CRYS_FLOW_HMAC);

		/* initialize HMAC engine data */
		CRYS_ATP_TST_FLOW_InitHMACEngineData(TST_index);

		/* Initialize HMAC flow */
		/************************/
		TST_ATP_Error = CRYS_FLOW_HMACInit(CRYS_FLOW_TypeDynamic,
			                               &TST_FLOWParamsInfo,
			                               &TST_FLOWHMACEngineData,  
			                               &flowId);

	
		ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_HMACInit is done\n"));

		TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_HMACInit",TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif  

	    /* Send interrupts that input and output buffers are ready */
        /***********************************************************/
        
		/* Send interrupt that input buffers are ready */
        CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);

		/* initialize buffers to add to flow */
		inputBuffersArraySize  = TST_HMACFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
		
        /* prepare buffers to add */
	    for (j = 0; j < inputBuffersArraySize - 1; ++j )
		{
			/* point to the next block of the input buffer */
			inputBuffersArray_ptr[j].buffer_ptr = (DxUint8_t*)TST_FLOW_InputData + (j+1)*blockSize;

			inputBuffersArray_ptr[j].bufferSize = blockSize;                
		}

		/* Add buffers to dynamic flow */
            /*******************************/
		TST_ATP_Error = CRYS_FLOW_AddBuffers(flowId,
                				             inputBuffersArray_ptr,
                				             inputBuffersArraySize - 1,
                				             DX_NULL,
                				             0);
                				             
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_AddBuffers is done\n"));

		TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_AddBuffers",TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif                  				             
				
        /* Send interrupts that added input and output buffers are ready */
        /*****************************************************************/
            
        /* Send interrupt that input buffers are ready */
        for (j = 0; j < inputBuffersArraySize - 1; ++j )
            CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
                
        blockSize = 0;    
       

		/* Finish HMAC flow, get results and compare with expected one */
		/********************/
		TST_TestStatus = CRYS_ATP_TST_FinishHMACFlow(TST_index, flowId, TST_MasterTestName, 
 	                                                 blockSize, &TST_FLOWParamsInfo);
 	                                             
        #if CRYS_ATP_TST_EXIT_ON_ERROR
    	if (TST_TestStatus != TESTPASS) 
    		return TESTFAIL;
        #endif  
    
        /* terminate flow */
        /******************/
        TST_ATP_Error = CRYS_FLOW_Terminate(flowId);
		
		ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Terminate is done\n")); 
		
		TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Terminate",TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif  

	}/* End of TST_index */     

	
	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_HMACDYNAMIC_AddBuffers_Acceptance*/
 

/*****************************************************************************
* Function Name:                                                           
*  CRYS_ATP_TST_FLOW_HMAC_ONETEST_Acceptance_WithAdditionalEngines
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for HMAC FLOW API's with additional engines.

* Algorithm:  
*  1. Init HMAC flow with type and other parameters
*  2. Send interrupts that input and output buffers are ready
*  3. Execute some AES operation and compare the result to expected
*  4. Send several interrupts that input and output buffers are ready, 
*  5. Execute some HASH operation and compare the result to expected
*  6. Send several interrupts that input and output buffers are ready, 
*  7. Execute some DES operation and compare the result of the AES to expected
*  8. Send several interrupts that input and output buffers are ready, 
*  9. Finish HMAC flow, get result and compare with expected one
******************************************************************************/
CRYSError_t CRYS_ATP_TST_FLOW_HMAC_ONETEST_Acceptance_WithAdditionalEngines ( 
                                                                int              TST_index, 
                                                                char             *TST_MasterTestName,
                                                                CRYS_FLOW_Type_t flowType ) 
{

	/********* TST Variables Definitions ************/
    CRYSError_t             	 TST_TestStatus = TESTPASS;
	/* flow id */
	DxUint32_t                   flowId = 0;
	DxUint16_t                   blockSize = 0;
	DxUint32_t                   i = 0;
	
	
	/************************************************/
	/*********** Start of test **********************/
	/************************************************/
    TST_TestData.inputBigEndianOrderFlag  = TST_HMACFLOW_Acceptance_TestData[TST_index].inputBigEndianOrderFlag;
	TST_TestData.inputBuffersArraySize    = TST_HMACFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
	TST_TestData.inputInterruptId         = TST_HMACFLOW_Acceptance_TestData[TST_index].inputInterruptId;
	TST_TestData.outputBigEndianOrderFlag = TST_HMACFLOW_Acceptance_TestData[TST_index].outputBigEndianOrderFlag;
	TST_TestData.outputBuffersArraySize   = TST_HMACFLOW_Acceptance_TestData[TST_index].outputBuffersArraySize;
	TST_TestData.outputInterruptId        = TST_HMACFLOW_Acceptance_TestData[TST_index].outputInterruptId;
	TST_TestData.wordsBuffersOrderFlag    = TST_HMACFLOW_Acceptance_TestData[TST_index].wordsBuffersOrderFlag;

	/* input block size */
	blockSize       = TST_FLOW_InputDataSize/TST_HMACFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
	
	/* prepare input buffer: */
	DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_Input_Data, TST_FLOW_InputDataSize);
	
	/* Initialize HMAC flow parameters and init HMAC flow by calling function */
	/* CRYS_FLOW_HMACInit                                                     */
	/**************************************************************************/
	TST_TestStatus = CRYS_ATP_TST_InitHMACFlow(TST_index, flowType, TST_MasterTestName, &flowId, &TST_FLOWParamsInfo);
	
	#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
    #endif  

	/* Send interrupt that input buffer is ready */
	/***********************************************************/
    
	/* Send interrupt that input buffers are ready */
	CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
	
	/* execute AES operation and compare with expected result */
	/**********************************************************/
	TST_ATP_Error = CRYS_ATP_TST_FLOW_CheckAESOperation();
	
	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"AES operation",TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  
	
	/* Send interrupt that input buffer is ready */
	/*********************************************/
	/* Send interrupt that input buffers are ready */
	CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
	
	/* execute HASH operation and compare with expected result */
	/***********************************************************/
	TST_ATP_Error = CRYS_ATP_TST_FLOW_CheckHASHOperation();
	

	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"HASH operation",TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  

    /* Send interrupt that input buffer is ready */
	/*********************************************/
    /* Send interrupt that input buffers are ready */
	CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
	
    /* execute DES operation and compare with expected result */
	/***********************************************************/
    TST_ATP_Error = CRYS_ATP_TST_FLOW_CheckDESOperation();
    
    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"DES operation",TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  


    /* Send interrupt that input buffers are ready */
	/*********************************************/
	
    /* Send interrupt that input buffers are ready */
    if ((TST_FLOW_IsFinishSizeZero == DX_TRUE) ||
        (TST_FLOWParamsInfo.inputBuffersArraySize < 4))
    {
        /* inputBuffersArraySize should be minimum 3 */
        for (i = 0; i < TST_FLOWParamsInfo.inputBuffersArraySize - 3; ++i)
	    {
	        CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
	    }
	    
	    blockSize = 0;
    }
    else
    {
        for (i = 0; i < TST_FLOWParamsInfo.inputBuffersArraySize - 4; ++i)
	    {
	        CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
	    }
    }
	

	/* Finish HMAC flow, get results and compare results with expected  */
    /* by calling functions CRYS_FLOW_Finish and CRYS_FLOW_GetResults   */
    /********************************************************************/
	TST_TestStatus = CRYS_ATP_TST_FinishHMACFlow(TST_index, flowId, TST_MasterTestName, 
 	                                             blockSize, &TST_FLOWParamsInfo);
 	                                             
    #if CRYS_ATP_TST_EXIT_ON_ERROR
    if (TST_TestStatus != TESTPASS) 
        return TESTFAIL;
    #endif  
    
    
    /* Terminate flow */
    /******************/
    TST_ATP_Error = CRYS_FLOW_Terminate(flowId);
	
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Terminate is done\n"));                             
	
	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Terminate",TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  

	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_HMAC_ONETEST_Acceptance_WithAdditionalEngines*/ 

/*****************************************************************************
* Function Name:                                                           
*  CRYS_ATP_TST_FLOW_HMACSTATIC_Acceptance_WithAdditionalEngines
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for static HMAC FLOW API's with additional engines.

* Algorithm:     
*  This function goes through all flow vectors and execute acceptance test with 
*  additional engines for each vector
******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_HMACSTATIC_Acceptance_WithAdditionalEngines ( void ) 
{

	/********* TST Variables Definitions ************/
	int                     	 TST_index;
	CRYSError_t             	 TST_TestStatus = TESTPASS;
	char                     	 TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_HMACSTATIC_Acceptance_WithAdditionalEngines"; 


	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;


	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"******************************************************************************************* \n"));
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_HMACSTATIC_Acceptance_WithAdditionalEngines Tests ********************* \n"));      
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"******************************************************************************************* \n \n"));


	/* go through all flow vectors and execute acceptance test */
	/***********************************************************/
	for (TST_index =0 ; TST_index < TST_NumberOf_HMACFLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
	{
		TST_TestStatus = CRYS_ATP_TST_FLOW_HMAC_ONETEST_Acceptance_WithAdditionalEngines(
		                                                           TST_index,
	                                                               TST_MasterTestName,
	                                                               CRYS_FLOW_TypeStatic);
	                                                               
#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif  
	                                                               
	}/* End of TST_index */     

	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_HMACSTATIC_Acceptance_WithAdditionalEngines*/ 


/*****************************************************************************
* Function Name:                                                           
*  CRYS_ATP_TST_FLOW_HMACDYNAMIC_Acceptance_WithAdditionalEngines
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for static HMAC FLOW API's with additional engines.

* Algorithm:     
*  This function goes through all flow vectors and execute acceptance test with 
*  additional engines for each vector
******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_HMACDYNAMIC_Acceptance_WithAdditionalEngines ( void ) 
{

	/********* TST Variables Definitions ************/
	int                     	 TST_index;
	CRYSError_t             	 TST_TestStatus = TESTPASS;
	char                     	 TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_HMACDYNAMIC_Acceptance_WithAdditionalEngines"; 
	
	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;


	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"******************************************************************************************* \n"));
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_HMACDYNAMIC_Acceptance_WithAdditionalEngines Tests ********************* \n"));      
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"******************************************************************************************* \n \n"));


	/* go through all flow vectors and execute acceptance test */
	/***********************************************************/
	for (TST_index =0 ; TST_index < TST_NumberOf_HMACFLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
	{
		TST_TestStatus = CRYS_ATP_TST_FLOW_HMAC_ONETEST_Acceptance_WithAdditionalEngines(
		                                                           TST_index,
	                                                               TST_MasterTestName,
	                                                               CRYS_FLOW_TypeDynamic);
	                                                               
#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif  
	                                                               
	}/* End of TST_index */     

	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_HMACDYNAMIC_Acceptance_WithAdditionalEngines*/ 

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_HMACSTATIC_ONETEST_Cyclic_Acceptance
 *  
 * Inputs:
 *  TST_index          - test index
 *  TST_MasterTestName - test name 
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for static cyclic HMAC FLOW API's
 
 * Algorithm:     
 * 1. Initialize HMAC flow with static type and other parameters 
 * 2. Init HMAC flow
 * 3. Send interrupts to execute cyclic flow operation 
 * 4. Finish HMAC flow
 * 5. Get results
 * 6. Terminate flow
 * 7. Read data from the output buffers and compare to expected result
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_HMACSTATIC_ONETEST_Cyclic_Acceptance ( int   TST_index,
                                                                     char *TST_MasterTestName  ) 
{

	/********* TST Variables Definitions ************/
	CRYSError_t             	 TST_TestStatus = TESTPASS;
	
	/* flow id */
	DxUint32_t                   flowId = 0;
	int                          i = 0;

	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;

	
	TST_TestData.inputBigEndianOrderFlag  = TST_HMACFLOW_Acceptance_TestData[TST_index].inputBigEndianOrderFlag;
	TST_TestData.inputBuffersArraySize    = TST_HMACFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
	TST_TestData.inputInterruptId         = TST_HMACFLOW_Acceptance_TestData[TST_index].inputInterruptId;
	TST_TestData.outputBigEndianOrderFlag = TST_HMACFLOW_Acceptance_TestData[TST_index].outputBigEndianOrderFlag;
	TST_TestData.outputBuffersArraySize   = TST_HMACFLOW_Acceptance_TestData[TST_index].outputBuffersArraySize;
	TST_TestData.outputInterruptId        = TST_HMACFLOW_Acceptance_TestData[TST_index].outputInterruptId;
	TST_TestData.wordsBuffersOrderFlag    = TST_HMACFLOW_Acceptance_TestData[TST_index].wordsBuffersOrderFlag;

	/* prepare input buffer: */
	DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_Input_Data, TST_FLOW_InputDataSize);
	
	/* Initialize HMAC flow data */
    /****************************/
	CRYS_ATP_TST_FLOW_InitParams(TST_index, DX_TRUE, &TST_TestData, &TST_FLOWParamsInfo, CRYS_FLOW_HMAC);
	
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;

	/* initialize HMAC engine data */
	CRYS_ATP_TST_FLOW_InitHMACEngineData(TST_index);


	/* Initialize HMAC flow */
	/************************/
	TST_ATP_Error = CRYS_FLOW_HMACInit(CRYS_FLOW_TypeStatic,
			&TST_FLOWParamsInfo,
			&TST_FLOWHMACEngineData,  
			&flowId);

	
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_HMACInit is done\n"));

	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_HMACInit",TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  
    
    /* Send interrupts to execute cyclic flow operation */
    /****************************************************/

	/* Send interrupt that input buffers are ready */
	for (i = 0; i < TST_FLOWParamsInfo.inputBuffersArraySize; ++i)
        CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
	
	
    /* Send interrupt that input buffers are ready  second time */
    for (i = 0; i < TST_FLOWParamsInfo.inputBuffersArraySize; ++i)
        CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
	
	
    /* Finish HMAC flow with size of the last input buffer */
	/******************************************************/
    CRYS_FLOW_Finish(flowId,0);  

	ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Finish is done\n"));                             
	
	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Finish",TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif

	
	/* Get results */
	/***************/
	do
    {
        
        TST_ATP_Error = CRYS_FLOW_GetResults(flowId,
		                                 (DxUint8_t*)TST_FLOW_Hash_OutputData,
		                                 CRYS_HASH_RESULT_SIZE_IN_WORDS*4);
	
    }while(TST_ATP_Error == DX_SEP_FLOW_GET_RESULT_NOT_READY_ERROR);
	

	ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_GetResults is done\n"));
	
	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_GetResults",TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  
	
	/* Terminate flow */
    /******************/
	CRYS_FLOW_Terminate(flowId);
	
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Terminate is done\n"));                             
        
    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Terminate",TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
 	
    #if CRYS_ATP_TST_EXIT_ON_ERROR
    if (TST_TestStatus != TESTPASS) 
        return TESTFAIL;
    #endif  

	/* Compare otuput to expected result */
	/*************************************/ 
	
	/* compare and prints results of test */
	TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(
		(DxUint8_t*)TST_FLOW_Hash_OutputData,
		TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Expected_cyclicOperation_data,
	    TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Expected_dataSize);


	/*Print the outputted and then the expected result after mem compare */
	ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
		"\nData outputted by engine [output buffer]-- \n",
			(DxUint8_t*)TST_FLOW_Hash_OutputData, 
		    TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Expected_dataSize); 

	ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
		"\nData Expected by engine --\n",
		TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Expected_output_data,
		TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Expected_dataSize);                 	                                     			
	

	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Name,TST_HMACFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  

	return TST_TestStatus;
 }/* End of CRYS_ATP_TST_FLOW_HMACSTATIC_ONETEST_Cyclic_Acceptance*/ 


/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_HMACSTATIC_Cyclic_Acceptance
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for static cyclic HMAC FLOW API's
 
 * Algorithm:     
 *	This function initializes HMAC flow with static type and other parameters,
 * and executes several times flow on the same input buffer. It goes throw all test
 * vectors and check cyclic acceptance.
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_HMACSTATIC_Cyclic_Acceptance ( void ) 
{

	/********* TST Variables Definitions ************/
	int                     	 TST_index;
	CRYSError_t             	 TST_TestStatus = TESTPASS;
	char                     	 TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_HMACSTATIC_Cyclic_Acceptance"; 
	

	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;
	
	
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n"));
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_HMACSTATIC_Cyclic_Acceptance Tests ********************* \n"));      
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n \n"));


	/* This function goes throw all test vectors and check cyclic acceptance */
	/*************************************************************************/
	for (TST_index =0 ; TST_index < TST_NumberOf_HMACFLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
    {
     	
     	TST_TestStatus = CRYS_ATP_TST_FLOW_HMACSTATIC_ONETEST_Cyclic_Acceptance(TST_index, TST_MasterTestName);
     	
#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
		    return TESTFAIL;
#endif  		                                                               

	                            
	}/* End of TST_index */     
    
  	return TST_TestStatus;
 }/* End of CRYS_ATP_TST_FLOW_HMACSTATIC_Acceptance_WithAdditionalEngines*/ 

 

#endif
